"use strict";
const tslint = require("tslint");
const path = require("path");
//TODO we "steal"" an error code with a registered code fix. 2515 = implement inherited abstract class
const TSLINT_ERROR_CODE = 2515;
function init(modules) {
    const ts = modules.typescript;
    let codeFixActions = new Map();
    let registeredCodeFixes = false;
    let configCache = {
        filePath: null,
        configuration: null,
        isDefaultConfig: false,
        configFilePath: null
    };
    // Work around the lack of API to register a CodeFix
    function registerCodeFix(action) {
        return ts.codefix.registerCodeFix(action);
    }
    if (!registeredCodeFixes && ts && ts.codefix) {
        registerCodeFixes(registerCodeFix);
        registeredCodeFixes = true;
    }
    function registerCodeFixes(registerCodeFix) {
        // Code fix for that is used for all tslint fixes
        registerCodeFix({
            errorCodes: [TSLINT_ERROR_CODE],
            getCodeActions: (_context) => {
                return null;
            }
        });
    }
    function fixRelativeConfigFilePath(config, projectRoot) {
        if (!config.configFile) {
            return config;
        }
        if (path.isAbsolute(config.configFile)) {
            return config;
        }
        config.configFile = path.join(projectRoot, config.configFile);
        return config;
    }
    function create(info) {
        info.project.projectService.logger.info("tslint-language-service loaded");
        let config = fixRelativeConfigFilePath(info.config, info.project.getCurrentDirectory());
        let configuration = null;
        // Set up decorator
        const proxy = Object.create(null);
        const oldLS = info.languageService;
        for (const k in oldLS) {
            proxy[k] = function () {
                return oldLS[k].apply(oldLS, arguments);
            };
        }
        // key to identify a rule failure
        function computeKey(start, end) {
            return `[${start},${end}]`;
        }
        function makeDiagnostic(problem, file) {
            let message = (problem.getRuleName() !== null)
                ? `${problem.getFailure()} (${problem.getRuleName()})`
                : `${problem.getFailure()}`;
            let category;
            if (config.alwaysShowRuleFailuresAsWarnings === true) {
                category = ts.DiagnosticCategory.Warning;
            }
            else if (problem.getRuleSeverity && problem.getRuleSeverity() === 'error') {
                // tslint5 supports to assign severities to rules
                category = ts.DiagnosticCategory.Error;
            }
            else {
                category = ts.DiagnosticCategory.Warning;
            }
            let diagnostic = {
                file: file,
                start: problem.getStartPosition().getPosition(),
                length: problem.getEndPosition().getPosition() - problem.getStartPosition().getPosition(),
                messageText: message,
                category: category,
                source: 'tslint',
                code: TSLINT_ERROR_CODE
            };
            return diagnostic;
        }
        /**
         * Filter failures for the given document
         */
        function filterProblemsForDocument(documentPath, failures) {
            let normalizedPath = path.normalize(documentPath);
            // we only show diagnostics targetting this open document, some tslint rule return diagnostics for other documents/files
            let normalizedFiles = new Map();
            return failures.filter(each => {
                let fileName = each.getFileName();
                if (!normalizedFiles.has(fileName)) {
                    normalizedFiles.set(fileName, path.normalize(fileName));
                }
                return normalizedFiles.get(fileName) === normalizedPath;
            });
        }
        function replacementsAreEmpty(fix) {
            // in tslint 4 a Fix has a replacement property witht the Replacements
            if (fix.replacements) {
                return fix.replacements.length === 0;
            }
            // tslint 5
            if (Array.isArray(fix)) {
                return fix.length === 0;
            }
            return false;
        }
        function recordCodeAction(problem, file) {
            let fix = null;
            // tslint can return a fix with an empty replacements array, these fixes are ignored
            if (problem.getFix && problem.getFix() && !replacementsAreEmpty(problem.getFix())) {
                fix = problem.getFix(); // createAutoFix(problem, document, problem.getFix());
            }
            if (!fix) {
                return;
            }
            let documentAutoFixes = codeFixActions.get(file.fileName);
            if (!documentAutoFixes) {
                documentAutoFixes = new Map();
                codeFixActions.set(file.fileName, documentAutoFixes);
            }
            documentAutoFixes.set(computeKey(problem.getStartPosition().getPosition(), problem.getEndPosition().getPosition()), problem);
        }
        function getConfigurationFailureMessage(err) {
            let errorMessage = `unknown error`;
            if (typeof err.message === 'string' || err.message instanceof String) {
                errorMessage = err.message;
            }
            return `tslint: Cannot read tslint configuration - '${errorMessage}'`;
        }
        function getConfiguration(filePath, configFileName) {
            if (configCache.configuration && configCache.filePath === filePath) {
                return configCache.configuration;
            }
            let isDefaultConfig = false;
            let configuration;
            let configFilePath = null;
            isDefaultConfig = tslint.Configuration.findConfigurationPath(configFileName, filePath) === undefined;
            let configurationResult = tslint.Configuration.findConfiguration(configFileName, filePath);
            // between tslint 4.0.1 and tslint 4.0.2 the attribute 'error' has been removed from IConfigurationLoadResult
            // in 4.0.2 findConfiguration throws an exception as in version ^3.0.0
            if (configurationResult.error) {
                throw configurationResult.error;
            }
            configuration = configurationResult.results;
            // In tslint version 5 the 'no-unused-variable' rules breaks the TypeScript language service plugin.
            // See https://github.com/Microsoft/TypeScript/issues/15344
            // Therefore we remove the rule from the configuration.
            //
            // In tslint 5 the rules are stored in a Map, in earlier versions they were stored in an Object
            if (config.disableNoUnusedVariableRule === true || config.disableNoUnusedVariableRule === undefined) {
                if (configuration.rules && configuration.rules instanceof Map) {
                    configuration.rules.delete('no-unused-variable');
                }
                if (configuration.jsRules && configuration.jsRules instanceof Map) {
                    configuration.jsRules.delete('no-unused-variable');
                }
            }
            configFilePath = configurationResult.path;
            configCache = {
                filePath: filePath,
                isDefaultConfig: isDefaultConfig,
                configuration: configuration,
                configFilePath: configFilePath
            };
            return configCache.configuration;
        }
        function captureWarnings(message) {
            // TODO log to a user visible log and not only the TS-Server log
            info.project.projectService.logger.info(`[tslint] ${message}`);
        }
        function convertReplacementToTextChange(repl) {
            return {
                newText: repl.text,
                span: { start: repl.start, length: repl.length }
            };
        }
        function getReplacements(fix) {
            let replacements = null;
            // in tslint4 a Fix has a replacement property with the Replacements
            if (fix.replacements) {
                // tslint4
                replacements = fix.replacements;
            }
            else {
                // in tslint 5 a Fix is a Replacement | Replacement[]                  
                if (!Array.isArray(fix)) {
                    replacements = [fix];
                }
                else {
                    replacements = fix;
                }
            }
            return replacements;
        }
        function addRuleFailureFix(fixes, problem, fileName) {
            let fix = problem.getFix();
            let replacements = getReplacements(fix);
            fixes.push({
                description: `Fix '${problem.getRuleName()}'`,
                changes: [{
                        fileName: fileName,
                        textChanges: replacements.map(each => convertReplacementToTextChange(each))
                    }]
            });
        }
        function addDisableRuleFix(fixes, problem, fileName, file) {
            fixes.push({
                description: `Disable rule '${problem.getRuleName()}'`,
                changes: [{
                        fileName: fileName,
                        textChanges: [{
                                newText: `// tslint:disable-next-line:${problem.getRuleName()}\n`,
                                span: { start: file.getLineStarts()[problem.getStartPosition().getLineAndCharacter().line], length: 0 }
                            }]
                    }]
            });
        }
        function addOpenConfigurationFix(fixes) {
            // the Open Configuration code action is disabled since there is no specified API to open an editor
            let openConfigFixEnabled = false;
            if (openConfigFixEnabled && configCache && configCache.configFilePath) {
                fixes.push({
                    description: `Open tslint.json`,
                    changes: [{
                            fileName: configCache.configFilePath,
                            textChanges: []
                        }]
                });
            }
        }
        function addAllAutoFixable(fixes, documentFixes, fileName) {
            const allReplacements = getNonOverlappingReplacements(documentFixes);
            fixes.push({
                description: `Fix all auto-fixable tslint failures`,
                changes: [{
                        fileName: fileName,
                        textChanges: allReplacements.map(each => convertReplacementToTextChange(each))
                    }]
            });
        }
        function getReplacement(failure, at) {
            return getReplacements(failure.getFix())[at];
        }
        function sortFailures(failures) {
            // The failures.replacements are sorted by position, we sort on the position of the first replacement
            return failures.sort((a, b) => {
                return getReplacement(a, 0).start - getReplacement(b, 0).start;
            });
        }
        function getNonOverlappingReplacements(documentFixes) {
            function overlaps(a, b) {
                return a.end >= b.start;
            }
            let sortedFailures = sortFailures([...documentFixes.values()]);
            let nonOverlapping = [];
            for (let i = 0; i < sortedFailures.length; i++) {
                let replacements = getReplacements(sortedFailures[i].getFix());
                if (i === 0 || !overlaps(nonOverlapping[nonOverlapping.length - 1], replacements[0])) {
                    nonOverlapping.push(...replacements);
                }
            }
            return nonOverlapping;
        }
        proxy.getSemanticDiagnostics = (fileName) => {
            const prior = oldLS.getSemanticDiagnostics(fileName);
            if (config.supressWhileTypeErrorsPresent && prior.length > 0) {
                return prior;
            }
            try {
                info.project.projectService.logger.info(`Computing tslint semantic diagnostics...`);
                if (codeFixActions.has(fileName)) {
                    codeFixActions.delete(fileName);
                }
                if (config.ignoreDefinitionFiles === true && fileName.endsWith('.d.ts')) {
                    return prior;
                }
                try {
                    configuration = getConfiguration(fileName, config.configFile);
                }
                catch (err) {
                    // TODO: show the reason for the configuration failure to the user and not only in the log
                    // https://github.com/Microsoft/TypeScript/issues/15913
                    info.project.projectService.logger.info(getConfigurationFailureMessage(err));
                    return prior;
                }
                let result;
                // tslint writes warning messages using console.warn()
                // capture the warnings and write them to the tslint plugin log
                let warn = console.warn;
                console.warn = captureWarnings;
                try {
                    // TODO the types of the Program provided by tsserver libary are not compatible with the one provided by typescript
                    // casting away the type
                    let options = { fix: false };
                    let linter = new tslint.Linter(options, oldLS.getProgram());
                    linter.lint(fileName, "", configuration);
                    result = linter.getResult();
                }
                catch (err) {
                    let errorMessage = `unknown error`;
                    if (typeof err.message === 'string' || err.message instanceof String) {
                        errorMessage = err.message;
                    }
                    info.project.projectService.logger.info('tslint error ' + errorMessage);
                    return prior;
                }
                finally {
                    console.warn = warn;
                }
                if (result.failures.length > 0) {
                    const tslintProblems = filterProblemsForDocument(fileName, result.failures);
                    if (tslintProblems && tslintProblems.length) {
                        const file = oldLS.getProgram().getSourceFile(fileName);
                        const diagnostics = prior ? [...prior] : [];
                        tslintProblems.forEach(problem => {
                            diagnostics.push(makeDiagnostic(problem, file));
                            recordCodeAction(problem, file);
                        });
                        return diagnostics;
                    }
                }
            }
            catch (e) {
                info.project.projectService.logger.info(`tslint-language service error: ${e.toString()}`);
                info.project.projectService.logger.info(`Stack trace: ${e.stack}`);
            }
            return prior;
        };
        proxy.getCodeFixesAtPosition = function (fileName, start, end, errorCodes, formatOptions) {
            let prior = oldLS.getCodeFixesAtPosition(fileName, start, end, errorCodes, formatOptions);
            if (config.supressWhileTypeErrorsPresent && prior.length > 0) {
                return prior;
            }
            info.project.projectService.logger.info("tslint-language-service getCodeFixes " + errorCodes[0]);
            let documentFixes = codeFixActions.get(fileName);
            if (documentFixes) {
                const fixes = prior ? [...prior] : [];
                let problem = documentFixes.get(computeKey(start, end));
                if (problem) {
                    addRuleFailureFix(fixes, problem, fileName);
                }
                addAllAutoFixable(fixes, documentFixes, fileName);
                if (problem) {
                    addOpenConfigurationFix(fixes);
                    addDisableRuleFix(fixes, problem, fileName, oldLS.getProgram().getSourceFile(fileName));
                }
                return fixes;
            }
            return prior;
        };
        return proxy;
    }
    return { create };
}
module.exports = init;
//# sourceMappingURL=data:application/json;base64,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