#!/usr/bin/env python3

from flask import Flask, make_response, url_for, request, render_template, Markup
from flask_mysqldb import MySQL
from secretarchive import api, api_file, db, login_archive, profile_archive, registration, utils
import secretarchive

app = Flask(__name__)

app.config['MYSQL_USER'] = 'root'
app.config['MYSQL_PASSWORD'] = '1amR00t!'
app.config['MYSQL_DB'] = 'secretarchive'
app.config['MYSQL_HOST'] = 'db'  # IP address of Docker service
app.config['MYSQL_PORT'] = 3306

app.config['UPLOAD_FOLDER'] = 'secretfiles'
app.config['COOKIE_NAME']   = 'secret_session'

# app.config['PASSWORD_SALT_LENGTH'] = 24

mysql = MySQL(app)


@app.route('/', methods=['GET'])
def root():
    if utils.check_session(request=request, mysql=mysql) is None:
        active_session = False
    else:
        active_session = True

    if active_session:
        msg = "You can logout, show your profile, or upload all the data ;)"
    else:
        msg = "Is this your first time here? Just register and upload all the data (;"

    return render_template('index.html',
                           msg=Markup.escape(msg),
                           active_session=active_session)


@app.route('/register', methods=['GET', 'POST'])
def register():
    if request.method == 'GET':
        return registration.provide_form()
    else:
        print("request.form:", flush=True)
        print(request.form, flush=True)

        # 0. check input
        if request.form is None:
            return registration.provide_form("No form data provided!")

        name = request.form['name']
        pwd  = request.form['pwd']
        info = request.form['info']

        if not utils.validate_input(name, pwd):
            return registration.provide_form("Form data was not valid!")

        # 1. does the user already exist?
        if db.user_exists(mysql, name):
            return registration.provide_form("Username already exists!")

        # 2. create user
        return registration.handle(mysql, name, pwd, info)


@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'GET':
        return login_archive.provide_form()
    else:
        print("request.form:", flush=True)
        print(request.form, flush=True)

        # 0. check input
        if request.form is None:
            return login_archive.provide_form("No form data provided!")

        name = request.form['name']
        pwd  = request.form['pwd']

        if not utils.validate_input(name, pwd):
            return login_archive.provide_form("Form data was not valid!")

        # 1. handle login
        return login_archive.handle(mysql, name, pwd)


@app.route('/logout', methods=['GET', 'POST'])
def logout():
    # 0. login required
    user_id = utils.check_session(request=request, mysql=mysql)
    if user_id is None:
        return "Well, you need to login first to be able to logout. Nice try ;)"

    # 1. logout
    cookie = utils.generate_cookie(id='', token='')
    response = make_response(render_template('index.html',
                                             msg=Markup.escape("Logged out!"),
                                             active_session=False))

    response.set_cookie(app.config['COOKIE_NAME'], cookie, httponly=True)
    return response


# Endpoint for testing
@app.route('/hi/<int:id>', methods=['GET'])
def hello_name(id):
    return "Hello %d :D" % id


@app.route('/api/', methods=['GET'])
def provide_api():
    # 0. login required
    user_id = utils.check_session(request=request, mysql=mysql)
    if user_id is None:
        return login_archive.provide_form("Please login first!")

    # 1. display API
    return api.provide_form(active_session=True)


@app.route('/api/file', methods=['GET', 'POST'])
def provide_api_file():
    # 0. login required
    user_id = utils.check_session(request=request, mysql=mysql)
    if user_id is None:
        return login_archive.provide_form("Please login first!")

    # 1. handle request
    if request.method == 'GET':
        return api_file.provide_form(active_session=True)
    else:
        # 1. check if a file is present
        file = request.files['file']

        if file is None:
            return api_file.provide_form("Please provide a file!")

        return api_file.handle_upload(file=file, user_id=user_id, mysql=mysql,
                                      active_session=True)


@app.route('/api/file/<string:fname>', methods=['GET', 'POST'])
def get_file(fname):

    return api_file.provide_file(fname)


@app.route('/about', methods=['GET'])
def about():
    ret_str = '''
    "Wat? Exactly." - <a href='https://fuzzy.land/FAQ'>fuzzy.land</a>
    '''

    return ret_str


@app.route('/profile', methods=['GET'])
def profile():
    # 0. login required
    user_id = utils.check_session(request=request, mysql=mysql)
    if user_id is None:
        return login_archive.provide_form("Please login first!")

    # 1. show profile
    return profile_archive.handle(mysql, user_id)


with app.test_request_context():
    print(url_for('root'))
    print(url_for('root', param='query'))
    print(url_for('hello_name', id=2))
    print(url_for('hello_name', param='query', id=2))
    print(url_for('about'))
    print(url_for('register'))
    print(url_for('login'))


if __name__ == "__main__":

    app.run(debug=True, host="0.0.0.0", port=5555)
